'use strict';


var SupportCollectionName = 'siteSupportingVAR';
var HistoryName = 'patientHistoryRequired';


var TypesOfCare = {
    PRIMARY_CARE: '323',
    MENTAL_HEALTH: '502',
    AUDIOLOGY: '203',
    OPTOMETRY: '408'
};


var Status = {
    // No History is not required.  Therefore, yes
    ENABLED: 'No',
    // Intentionally empty
    DISABLED: ''
};


var Direct = {
    COLLECTION: 'directBookingEligibilityCriteria',
    SETTINGS: 'coreSettings',
};


var Request = {
    COLLECTION: 'requestEligibilityCriteria',
    SETTINGS: 'requestSettings',
    CUSTOM: 'customRequestSettings'
};


function _createObject(key, data) {
    var obj = {};
    obj[key] = data;
    return obj;
}

function _get(connection, id) {
    var cursor = connection.find({_id: id});
    return cursor.toArray();
}


function _has(connection, id) {
    var results = _get(connection, id);
    return results.length === 1;
}


function _insert(connection, data) {
    return connection.insert(data);
}


function _update(connection, id, data) {
    return connection.update({_id: id}, data);
}


function supportingCollectionHasFacility(collection, id) {
    return _has(collection, id);
}


function _hasPatientHistory(connection, id, settingsKey, careType) {
    return !!_getPatientHistory(connection, id, settingsKey, careType);
}


function _getAllCareTypes(connection, id, settingsKey) {
    var result = _get(connection, id);
    if (result.length) {
        return result[0][settingsKey];
    }
    return null;
}


function _getCareType(connection, settingsKey, id, careType) {
    var careTypes = _getAllCareTypes(connection, settingsKey, id) || [];
    for (var i = 0; i < careType.length; i++) {
        var item = careTypes[i] || {};
        if (item._id === careType) {
            return item;
        }
    }
    return null;
}


function _getCareTypeIndex(connection, settingsKey, id, careType) {
    var careTypes = _getAllCareTypes(connection, settingsKey, id) || [];
    for (var i = 0; i < careTypes.length; i++) {
        var item = careTypes[i] || {};
        if (item._id === careType) {
            return i;
        }
    }
    return -1;
}


function _getPatientHistory(connection, id, settingsKey, careType) {
    var results = _getCareType(connection, id, settingsKey, careType);
    if (results) {
        return results[HistoryName];
    }
    return null;
}


function _insertFacility(connection, id, settingsKey) {
    var data = {_id: id};
    data[settingsKey] = [];
    if (settingsKey === Request.SETTINGS) {
        // Currently VAOS crashes without this (will be fixed in var-resources also)
        data[Request.CUSTOM] = [];
    }
    return _insert(connection, data);
}


function _insertTypeOfCare(connection, id, settingKey, careType) {
    var data = {_id: careType};
    var keyedData = _createObject(settingKey, data);
    var pushData = {$push: keyedData};
    return _update(connection, id, pushData);
}


function _insertPatientHistory(connection, id, settingsKey, careType, status) {
    var index = _getCareTypeIndex(connection, id, settingsKey, careType);
    var key = settingsKey + '.' + index + '.' + HistoryName;
    var keyedStatus = _createObject(key, status);
    var pushData = {$set: keyedStatus};
    return _update(connection, id, pushData);
}


function _updatePatientHistory(connection, id, settingsKey, careType, data) {
    var index = _getCareTypeIndex(connection, id, settingsKey, careType);
    var key = settingsKey + '.' + index + '.' + HistoryName;
    var keyedData = _createObject(key, data);
    var setData = {$set: keyedData};
    return _update(connection, id, setData);
}


function _destroyPatientHistory(connection, id, settingsKey, item) {
    var which = _createObject(settingsKey, item);
    var data = {$pull: which};
    return _update(connection, id, data);
}


function allowScheduling(connection, id, settingsKey, careType, status) {
    if (_has(connection, id)) {
        var index = _getCareTypeIndex(connection, id, settingsKey, careType);
        if (index > -1) {
            _updatePatientHistory(connection, id, settingsKey, careType, status);
        } else {
            _insertTypeOfCare(connection, id, settingsKey, careType);
            _insertPatientHistory(connection, id, settingsKey, careType, status);
        }
    } else {
        _insertFacility(connection, id, settingsKey);
        _insertTypeOfCare(connection, id, settingsKey, careType);
        _insertPatientHistory(connection, id, settingsKey, careType, status);
    }
}


function denyScheduling(connection, id, settingsKey) {
   var careTypes = _getAllCareTypes(connection, id, settingsKey) || [];
   careTypes.forEach(function(care) {
       var careType = care._id;
       _updatePatientHistory(connection, id, settingsKey, careType, Status.DISABLED);
   });
}


function removeExpressCare(connection, id) {
    var setData = {};
    setData[Request.CUSTOM] = [];

    _update(connection, id, { $set: setData });
}


function runner() {
    var facilities = getFacilities();
    var direct = db.getCollection(Direct.COLLECTION);
    var requests = db.getCollection(Request.COLLECTION);
    var supporting = db.getCollection(SupportCollectionName);
    var errors = [];

    facilities.forEach(function(id) {
        try {
            var parent = id;
            if (id.length === 5) {
                parent = id.substr(0,3);
                allowScheduling(direct, id, Direct.SETTINGS, TypesOfCare.AUDIOLOGY, Status.DISABLED);
                allowScheduling(direct, id, Direct.SETTINGS, TypesOfCare.OPTOMETRY, Status.DISABLED);
                allowScheduling(direct, id, Direct.SETTINGS, TypesOfCare.MENTAL_HEALTH, Status.DISABLED);
                allowScheduling(requests, id, Request.SETTINGS, TypesOfCare.MENTAL_HEALTH, Status.DISABLED);
                allowScheduling(requests, id, Request.SETTINGS, TypesOfCare.AUDIOLOGY, Status.DISABLED);
                allowScheduling(requests, id, Request.SETTINGS, TypesOfCare.OPTOMETRY, Status.DISABLED);

                removeExpressCare(requests, id);
            }

            if (supportingCollectionHasFacility(supporting, parent)) {
                allowScheduling(direct, id, Direct.SETTINGS, TypesOfCare.PRIMARY_CARE, Status.ENABLED);
                allowScheduling(requests, id, Request.SETTINGS, TypesOfCare.PRIMARY_CARE, Status.ENABLED);
                allowScheduling(requests, id, Request.SETTINGS, TypesOfCare.MENTAL_HEALTH, Status.ENABLED);
            } else {
                denyScheduling(direct, id, Direct.SETTINGS);
                denyScheduling(requests, id, Request.SETTINGS);

                removeExpressCare(requests, id);
            }
            print('Facility:', id, 'updated successfully');
        } catch (error) {
            var errorItem = {
                id: id,
                error: error
            };
            errors.push(errorItem);
        }
    });

    print();
    errors.forEach(function(error) {
        print('Error: ', error.id, 'failed to update. ', 'Message', error.error);
    });

    print('Process Complete');
}


function getFacilities() {
    // All valid "locations" as of 03/21/2018
    return [
        '358', '402', '402GA', '402GB', '402GC', '402GD', '402GE', '402GF', '402HB', '402HC', '405', '405GA',
        '405GC', '405HA', '405HE', '405HF', '436', '436A4', '436GA', '436GB', '436GC', '436GD', '436GF', '436GH',
        '436GI', '436GJ', '436GK', '436GL', '436GM', '436QA', '436QB', '436QC', '437', '437GA', '437GB', '437GC',
        '437GD', '437GE', '437GF', '437GI', '437GJ', '437GK', '437GL', '438', '438GA', '438GC', '438GD', '438GE',
        '438GF', '442', '442GB', '442GC', '442GD', '459', '459GA', '459GB', '459GC', '459GD', '459GE', '459GF',
        '459GG', '459GH', '460', '460GA', '460GC', '460GD', '460HE', '460HG', '463', '463GA', '463GB', '463GC',
        '463GD', '463GE', '501', '501G2', '501GA', '501GB', '501GC', '501GD', '501GE', '501GH', '501GI', '501GJ',
        '501GK', '501GM', '501GN', '501HB', '502', '502GA', '502GB', '502GE', '502GF', '502GG', '502QB', '503',
        '503GA', '503GB', '503GC', '503GD', '503GE', '504', '504HB', '506', '506GA', '506GB', '506GC', '508', '508GA',
        '508GE', '508GF', '508GG', '508GH', '508GI', '508GJ', '508GK', '508GL', '509', '509A0', '509GA', '509GB',
        '509QA', '512', '512A5', '512GA', '512GC', '512GD', '512GE', '512GF', '512GG', '515', '515BY', '515GA',
        '515GB', '515GC', '516', '516BZ', '516GA', '516GB', '516GC', '516GD', '516GE', '516GF', '516GH', '517', '517GB',
        '518', '518GA', '518GB', '518GE', '518GG', '519', '519GA', '519GB', '519GD', '519HC', '519HD', '519HF', '520',
        '520A0', '520GB', '520GC', '520QA', '521', '521GA', '521GB', '521GC', '521GD', '521GE', '521GF', '521GG',
        '521GH', '521GI', '521GJ', '523', '523A4', '523A5', '523GA', '523GB', '523GC', '523GD', '526', '526GA', '526GB',
        '526GD', '528', '528A4', '528A5', '528A6', '528A7', '528A8', '528G2', '528G3', '528G4', '528G5', '528G6',
        '528G7', '528G8', '528G9', '528GB', '528GC', '528GD', '528GE', '528GK', '528GL', '528GM', '528GN', '528GO',
        '528GP', '528GQ', '528GR', '528GT', '528GV', '528GW', '528GX', '528GY', '528GZ', '529', '529GA', '529GB',
        '529GC', '529GD', '529GF', '531', '531GE', '531GG', '531GH', '531GI', '531GJ', '534', '534BY', '534GB', '534GC',
        '534GD', '534GE', '534GF', '537', '537GA', '537GD', '537HA', '538', '538GA', '538GB', '538GC', '538GD', '538GE',
        '538GF', '539', '539GA', '539GB', '539GC', '539GD', '539GE', '539GF', '540', '540GA', '540GB', '540GC', '540GD',
        '541', '541BY', '541BZ', '541GB', '541GC', '541GD', '541GE', '541GF', '541GG', '541GH', '541GI', '541GJ',
        '541GK', '541GL', '541GM', '541GN', '542', '542GA', '542GE', '544', '544GB', '544GC', '544GD', '544GE', '544GF',
        '544GG', '546', '546GA', '546GB', '546GC', '546GD', '546GE', '546GF', '546GH', '548', '548GA', '548GB', '548GC',
        '548GD', '548GE', '548GF', '549', '549A4', '549BY', '549GA', '549GB', '549GC', '549GD', '549GE', '549GF',
        '549GH', '549GI', '549GJ', '549GK', '549GL', '550', '550GA', '550GD', '550GF', '552', '552GA', '552GB', '552GC',
        '552GD', '553', '553GA', '553GB', '554', '554GB', '554GC', '554GD', '554GE', '554GF', '554GG', '554GH', '554GI',
        '554QA', '556', '556GA', '556GC', '556GD', '557', '557GA', '557GB', '557GC', '557GE', '557GF', '558', '558GA',
        '558GB', '558GC', '558GD', '558GE', '558GF', '561', '561A4', '561BZ', '561GA', '561GB', '561GD', '561GE',
        '561GF', '561GH', '561GI', '561GJ', '561GK', '562', '562GA', '562GB', '562GC', '562GD', '562GE', '564', '564GA',
        '564GB', '564GC', '564GD', '564GE', '565', '565GA', '565GC', '565GD', '565GE', '565GF', '565GG', '565GH',
        '565GJ', '565GK', '565GL', '565GM', '565QD', '568', '568A4', '568GA', '568GB', '568HA', '568HB', '568HF',
        '568HH', '568HJ', '568HK', '568HP', '570', '570GA', '570GB', '570GC', '573', '573A4', '573GA', '573GD', '573GE',
        '573GG', '573GI', '573GJ', '573GK', '573GL', '573GM', '573GN', '575', '575GA', '578', '578GA', '578GC', '578GD',
        '578GE', '578GF', '578GG', '580', '580GC', '580GD', '580GE', '580GF', '580GG', '580GH', '581', '581GA', '581GB',
        '581GG', '581GH', '583', '583GA', '583GB', '583GC', '583GD', '583GE', '583QB', '585', '585GA', '585GB', '585GC',
        '585GD', '585HA', '585HB', '586', '586GA', '586GB', '586GC', '586GD', '586GE', '586GF', '586GG', '586QB',
        '586QC', '589', '589A4', '589A5', '589A6', '589A7', '589G1', '589G2', '589G3', '589G4', '589G5', '589G7',
        '589G8', '589G9', '589GB', '589GC', '589GD', '589GE', '589GF', '589GH', '589GI', '589GJ', '589GM', '589GN',
        '589GP', '589GR', '589GU', '589GV', '589GW', '589GX', '589GY', '589GZ', '589JA', '589JB', '589JC', '589JD',
        '589JE', '589JF', '590', '590GB', '590GC', '590GD', '593', '593GC', '593GH', '593QC', '595', '595GA', '595GC',
        '595GD', '595GE', '595GF', '595QA', '596', '596A4', '596GA', '596GB', '596GC', '596GD', '596HA', '598', '598A0',
        '598GA', '598GB', '598GC', '598GD', '598GE', '598GF', '598GG', '598GH', '600', '600GA', '600GB', '600GC',
        '600GD', '600GE', '603', '603GA', '603GB', '603GC', '603GD', '603GE', '603GF', '603GG', '603GH', '605', '605BZ',
        '605GA', '605GB', '605GC', '605GD', '605GE', '607', '607GC', '607GD', '607GE', '607GF', '607GG', '607HA', '608',
        '608GA', '608GC', '608GD', '608HA', '610', '610A4', '610GB', '610GC', '610GD', '612', '612A4', '612GD', '612GE',
        '612GF', '612GG', '612GH', '612GI', '612GJ', '612QD', '613', '613GA', '613GB', '613GC', '613GD', '613GE',
        '613GF', '613GG', '614', '614GA', '614GB', '614GC', '614GD', '614GE', '614GF', '614GG', '614GH', '614GI',
        '614GN', '618', '618GA', '618GB', '618GD', '618GE', '618GG', '618GH', '618GI', '618GJ', '618GK', '618GL',
        '618GM', '618GN', '619', '619A4', '619GA', '619GB', '619GD', '619GE', '619QB', '620', '620A4', '620GA', '620GB',
        '620GD', '620GE', '620GF', '620GG', '620GH', '621', '621BY', '621GA', '621GC', '621GG', '621GI', '621GJ',
        '621GK', '621QE', '623', '623BY', '623GA', '623GB', '626', '626A4', '626GA', '626GC', '626GE', '626GF', '626GG',
        '626GH', '626GJ', '626GK', '626GL', '626GM', '626GO', '626QC', '629', '629GA', '629GB', '629GC', '629GD',
        '629GE', '629GF', '630', '630A4', '630A5', '630GA', '630GB', '631', '631GC', '631GD', '631GE', '631GF', '631QB',
        '632', '632GA', '632HA', '632HB', '632HC', '632HD', '635', '635GB', '635GC', '635GD', '635GE', '635GF', '635GG',
        '635HB', '635QB', '636', '636A4', '636A5', '636A6', '636A8', '636GA', '636GB', '636GC', '636GD', '636GF',
        '636GG', '636GH', '636GI', '636GJ', '636GK', '636GL', '636GM', '636GN', '636GP', '636GQ', '636GR', '636GS',
        '636GT', '636GU', '636GV', '636GW', '636QI', '637', '637GA', '637GB', '637GC', '640', '640A0', '640A4', '640GA',
        '640GB', '640GC', '640HA', '640HB', '640HC', '642', '642GA', '642GC', '642GD', '642GF', '642QA', '644', '644BY',
        '644GA', '644GB', '644GC', '644GD', '644GE', '644GF', '644GG', '644GH', '646', '646A4', '646GA', '646GB',
        '646GC', '646GD', '646GE', '648', '648A4', '648GA', '648GB', '648GD', '648GE', '648GF', '648GG', '648GH',
        '648GI', '648GJ', '649', '649GA', '649GB', '649GC', '649GD', '649GE', '650', '650GA', '650GB', '650GD', '652',
        '652GA', '652GB', '652GE', '652GF', '652GG', '652GH', '653', '653BY', '653GA', '653GB', '653QA', '654', '654GA',
        '654GB', '654GC', '654GD', '654GE', '654QD', '655', '655GB', '655GC', '655GD', '655GE', '655GF', '655GG',
        '655GH', '655GI', '656', '656GA', '656GB', '656GC', '657', '657A0', '657A4', '657A5', '657GA', '657GB', '657GD',
        '657GF', '657GG', '657GH', '657GI', '657GJ', '657GK', '657GL', '657GM', '657GN', '657GO', '657GP', '657GQ',
        '657GR', '657GS', '657GT', '657GU', '657GV', '657GW', '657GX', '657GY', '658', '658GA', '658GB', '658GC',
        '658GD', '658GE', '659', '659BY', '659BZ', '659GA', '660', '660GA', '660GB', '660GC', '660GD', '660GE', '660GG',
        '660GJ', '662', '662GA', '662GC', '662GD', '662GE', '662GF', '662GG', '663', '663A4', '663GA', '663GB', '663GC',
        '663GD', '663GE', '664', '664GA', '664GB', '664GC', '664GD', '666', '666GB', '666GC', '666GD', '666GE', '666GF',
        '666QA', '666QB', '666QC', '667', '667GA', '667GB', '667GC', '668', '668GA', '668GB', '671', '671A4', '671GA',
        '671GC', '671GD', '671GF', '671GG', '671GH', '671GI', '671GJ', '671GK', '671GL', '671GN', '671GP', '671GQ',
        '672', '672GA', '672GB', '672GC', '672GD', '672GE', '673', '673GB', '673GC', '673GF', '674', '674A4', '674GA',
        '674GB', '674GC', '674GD', '674GF', '675', '675GA', '675GC', '675GD', '675GE', '675GF', '675GG', '676', '676GA',
        '676GC', '676GD', '676GE', '678', '678GA', '678GB', '678GC', '678GD', '678GE', '678GF', '678GG', '678QA',
        '678QB', '679', '679GA', '687', '687GA', '687GB', '687GC', '687HA', '688', '688GA', '688GB', '688GD', '688GE',
        '688QA', '689', '689A4', '689GA', '689GB', '689GC', '689GD', '689GE', '689HC', '691', '691A4', '691GB', '691GC',
        '691GD', '691GE', '691GF', '691GG', '691GK', '691GL', '691GM', '691GN', '691GO', '692', '692GA', '692GB', '693',
        '693GB', '693GC', '693GE', '693GF', '693GG', '695', '695GA', '695GC', '695GD', '740', '740GE', '740GH', '740GI',
        '740GJ', '756', '756GA', '756GB', '757', '757GA', '757GB', '757GC', '757GD'
    ].sort();
}


if (db._name !== 'var-utility') {
    print('Script must be executed on the var-utility database.');
} else {
    runner();
}
